/*
 * Object.h
 *
 * Created 7/31/2009 By Johnny Huynh
 *
 * Version 00.00.01 7/31/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef OBJECT_H
 #define OBJECT_H
 
 template <typename T> class Object;
 
 #include "pandaFramework.h"
 #include "referenceCount.h"
 #include "nodePathCollection.h"
 
 #include "global.h"
 #include "Vector.h"
 #include "Math3D.h"
 #include "typeHandle.h"
 #include "collisionEntry.h"
 #include "DamageObject.h"
 
 /**
  * Class specification for abstract class, Object (inanimate object)
  */
 template <typename T>
 class Object : public NodePath, public ReferenceCount
 {
 // Data Members
 private:
    //static TypeHandle _type_handle;
 protected:
 //NodePath node_path;
 
 // Local Functions
 public:
    Object();
    //Object( const NodePath& np );
    Object( const Object<T>& obj );
    virtual ~Object();
    inline Object<T>& operator=( const Object<T>& obj );
    virtual inline void add_task( GenericAsyncTask::TaskFunc* function, void* user_data );
    virtual inline void clear_tasks();
    virtual inline void clear_action_tasks();
    virtual inline DamageObject<T>* get_damage_object( const std::string& key );
    virtual inline void handle_from_collision( Object<T>& into, const CollisionEntry& c_entry );
    virtual inline void handle_into_collision( Object<T>& from, const CollisionEntry& c_entry );
    virtual inline void interrupt_action();
    virtual inline bool is_damage_object() const;
    virtual inline bool is_effected_by_gravity() const;
    virtual inline bool is_prone_to_effects() const;
    virtual inline void load_model( WindowFramework& window );
    virtual inline void load_model( WindowFramework& window, const NodePath& parent );
    virtual inline void load_model( WindowFramework& window, const std::string& filename );
    virtual inline void load_model( WindowFramework& window, const std::string& filename, const NodePath& parent );
    virtual inline void look_at_xy_direction( const T& x, const T& y );
    virtual inline void stun();
    virtual inline void unstun();
    virtual inline void translate_in_direction_of_heading( const T& heading_degrees, const T& units );
    virtual inline void translate( const T& x, const T& y );
    virtual inline void translate( const VECTOR2_TYPE& ground_direction );
    virtual inline void translate( const T& x, const T& y, const T& z );
    virtual inline void translate( const VECTOR3_TYPE& direction );
    
    // overloaded functions (NodePath)
    virtual void set_hpr(const LVecBase3f &hpr) { NodePath::set_h( Math3D::get_readjusted_angle_degrees( hpr.get_x() ) );
                                                  NodePath::set_p( Math3D::get_readjusted_angle_degrees( hpr.get_y() ) );
                                                  NodePath::set_r( Math3D::get_readjusted_angle_degrees( hpr.get_z() ) ); }
    virtual void set_h(float h) { NodePath::set_h( Math3D::get_readjusted_angle_degrees( h ) ); }
    virtual void set_p(float p) { NodePath::set_p( Math3D::get_readjusted_angle_degrees( p ) ); }
    virtual void set_r(float r) { NodePath::set_r( Math3D::get_readjusted_angle_degrees( r ) ); }
    static TypeHandle get_class_type() { return NodePath::get_class_type(); }
    static void init_type() { return NodePath::init_type(); }
	//static TypeHandle get_class_type() { return _type_handle; }
    //static void init_type() {
    //                            std::string template_type( typeid( T ).name() );
    //                            register_type(_type_handle, "Object<" + template_type + ">" );
    //                        }
	
 // Private Functions
 private:
    
 // Friend Functions
 public:
    
 };
 
 /** Static Data Members Initialization **/
 
 // Initialize static _type_handle
 //template <typename T> TypeHandle Object<T>::_type_handle;
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 Object<T>::Object()
           : NodePath()
 {
    
 }
 
 /**
  * Alternative Constructor
  */
 /*template <typename T>
 Object<T>::Object( const NodePath& np )
           : node_path( np )
 {
    
 }*/
 
 /**
  * Copy Constructor
  * Warning:
  * The Object copied will share the same NodePath information as this
  * Object due to PointerTo used in NodePath.
  */
 template <typename T>
 Object<T>::Object( const Object<T>& obj )
           : NodePath( static_cast< NodePath >( obj ) )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 Object<T>::~Object()
 {
    // remove all tasks associated with this Object
    this->clear_tasks();
    
    // Detach all NodePaths below this NodePath
    NodePathCollection collection( NodePath::get_children() );
    for ( size_t i( 0 ); i < collection.get_num_paths(); ++i )
    {
        NodePath node_path( collection.get_path( i ) );
        collection.add_paths_from( node_path.get_children() );
        node_path.detach_node();
    }
    
    // Detach this NodePath
    NodePath::detach_node();
 }
 
 /**
  * operator=() copies the content of the specified Object to this Object.
  * Warning:
  * The Object copied will share the same NodePath information as this
  * Object due to PointerTo used in NodePath.
  *
  * @param (const Object<T>& obj )
  * @return Object<T>&
  */
 template <typename T>
 inline Object<T>& Object<T>::operator=( const Object<T>& obj )
 {
    NodePath::operator=( static_cast< NodePath >( obj ) ); // because of internal pointerTo, this will not work as intended because pointing to same node
    return *this;
 }
 
 /**
  * May want to break into add_action_task and add_effect_task?
  * add_task() adds a newly created task, containing the specified function and user_data, to the global task manager.
  * The task is associated to this Object.
  *
  * @param (GenericAsyncTask::TaskFunc*) function
  * @param (void*) user_data
  */
 template <typename T>
 inline void Object<T>::add_task( GenericAsyncTask::TaskFunc* function, void* user_data )
 {
    global::_task_mgr_Ptr->add( new GenericAsyncTask(StringConversion::to_str( NodePath::get_key() ), function, user_data) );
 }
 
 /**
  * clear_tasks() removes all tasks in the global task manager that are associated with this Object.
  * It is assumed the name of the tasks associated with an Object is the Object's key.
  */
 template <typename T>
 inline void Object<T>::clear_tasks()
 {
    // clear action tasks associated with this Object
    Object<T>::clear_action_tasks();
    
    // clear global tasks associated with this Object
    AsyncTaskCollection task_collection( global::_task_mgr_Ptr->find_tasks( StringConversion::to_str(NodePath::get_key()) ) );
    size_t i( task_collection.size() );
    while ( i > 0 )
    {
        --i;
        global::_task_mgr_Ptr->remove( task_collection.get_task( i ) );
    }
 }
 
 /**
  * clear_action_tasks() removes this Object's current action (i.e. removes all 
  * tasks in the global action task manager that are associated with this Object).
  * It is assumed the name of the tasks associated with an Object is the Object's key.
  */
 template <typename T>
 inline void Object<T>::clear_action_tasks()
 {
    AsyncTaskCollection task_collection( global::_action_task_mgr_Ptr->find_tasks( StringConversion::to_str(NodePath::get_key()) ) );
    size_t i( task_collection.size() );
    while ( i > 0 )
    {
        --i;
        global::_action_task_mgr_Ptr->remove( task_collection.get_task( i ) );
    }
 }
 
 /**
  * get_damage_object() returns a pointer to the DamageObject matching the specified key.
  * If no DamageObject having the matching specified key is found, NULL is returned.
  *
  * @param (const std::string&) key
  * @return DamageObject<T>*
  */
 template <typename T>
 inline DamageObject<T>* Object<T>::get_damage_object( const std::string& key )
 {
    return NULL;
 }
 
 /**
  * handle_from_collision() handles the collision assuming this Object has
  * collided into another Object (additional information can be found 
  * inside of the CollisionEntry). That is, this Object is assumed to be
  * the "from" Object.
  *
  * @param (Object<T>&) into
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void Object<T>::handle_from_collision( Object<T>& into, const CollisionEntry& c_entry )
 {
    
 }
 
 /**
  * handle_into_collision() handles the collision assuming another Object
  * has collided into this Object (additional information can be found 
  * inside of the CollisionEntry). That is, this Object is assumed to be
  * the "into" Object.
  *
  * @param (Object<T>&) from
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void Object<T>::handle_into_collision( Object<T>& from, const CollisionEntry& c_entry )
 {
    
 }
 
 /**
  * interrupt_action() does not affect an Object.
  */
 template <typename T>
 inline void Object<T>::interrupt_action()
 {
    
 }
 
 /**
  * is_damage_object() returns true if this is a DamageObject; otherwise,
  * false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool Object<T>::is_damage_object() const
 {
    return false;
 }
 
 /**
  * is_effected_by_gravity() returns true if this Object is effected
  * by gravity; otherwise, false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool Object<T>::is_effected_by_gravity() const
 {
    return true;
 }
 
 
 /**
  * is_prone_to_effects() returns true if this Object is prone to 
  * physical or status effects; otherwise, false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool Object<T>::is_prone_to_effects() const
 {
    return false;
 }
 
 /**
  * load_model() does not load any Object model into the specified WindowFramework.
  * Use the load_model() functions of objects derived from this Object class for 
  * actual models.
  *
  * @param (WindowFramework&) window
  */
 template <typename T> 
 inline void Object<T>::load_model( WindowFramework& window )
 {
    
 }
 
 /**
  * load_model() does not load any Object model into the specified WindowFramework.
  * Use the load_model() functions of objects derived from this Object class for 
  * actual models.
  *
  * @param (WindowFramework&) window
  * @param (const NodePath&) parent
  */
 template <typename T> 
 inline void Object<T>::load_model( WindowFramework& window, const NodePath& parent )
 {
    
 }
 
 /**
  * load_model() loads the Object model at the location of the specified filename 
  * into the specified WindowFramework. To be visible, the Object needs to be reparented  
  * to a NodePath that is attached to the specified WindowFramework's scene graph.
  * The filename should be relative to the model path specified in the prc config 
  * file or the full path name of the file.
  *
  * @param (WindowFramework&) window
  * @param (const std::string&) filename
  */
 template <typename T> 
 inline void Object<T>::load_model( WindowFramework& window, const std::string& filename )
 {
    // We only load the model here
    NodePath::operator=( window.load_model( global::_framework.get_models(), filename ) );
 }
 
 /**
  * load_model() loads the Object model at the location of the specified filename 
  * into the specified WindowFramework. The Object is then parented to the specified
  * parent for rendering.
  * The filename should be relative to the model path specified in the prc config 
  * file or the full path name of the file.
  *
  * @param (WindowFramework&) window
  * @param (const std::string&) filename
  * @param (const NodePath&) parent
  */
 template <typename T> 
 inline void Object<T>::load_model( WindowFramework& window, const std::string& filename, const NodePath& parent )
 {
    // We only load the model here
    Object<T>::load_model( window, filename );
    
    // For the object to be visible, the NodePath must be reparented to the scene graph
    // Here, the NodePath is reparented to the specified parent
    NodePath::reparent_to( parent );
 }
 
 /**
  * look_at_xy_direction() reorients this object to face the specified
  * x and y direction. The specified vector does not have to be 
  * normalized.
  *
  * @param (const T&) x
  * @param (const T&) y
  */
 template <typename T>
 inline void Object<T>::look_at_xy_direction( const T& x, const T& y )
 {
    NodePath::set_h( Vector::get_angle_degrees( x, y ) );
    
    //look_at( get_x() + x, get_y() + y, get_z() );
 }
 
 /**
  * stun() does nothing; an Object cannot be stunned.
  */
 template <typename T>
 inline void Object<T>::stun()
 {
    
 }
 
 /**
  * unstun() does nothing; an Object cannot be stunned nor unstunned.
  */
 template <typename T>
 inline void Object<T>::unstun()
 {
    
 }
 
 /**
  * translate_in_direction_of_heading() translates this Object in the ground direction of the 
  * specified heading angle, measured in degrees, by the specified units.
  *
  * @param (const T&) heading_degrees
  * @param (const T&) units
  */
 template <typename T>
 inline void Object<T>::translate_in_direction_of_heading( const T& heading_degrees, const T& units )
 {
    Object<T>::translate( Vector::get_xy_direction_normalized(heading_degrees) * units );
 }
 
 /**
  * translate() translates this Object in the specified ground direction of the corresponding units.
  *
  * @param (const T&) x
  * @param (const T&) y
  */
 template <typename T>
 inline void Object<T>::translate( const T& x, const T& y )
 {
    NodePath::set_x( NodePath::get_x() + x );
    NodePath::set_y( NodePath::get_y() + y );
 }
 
 /**
  * translate() translates this Object in the specified ground direction of the corresponding units.
  *
  * @param (const VECTOR2_TYPE&) ground_direction
  */
 template <typename T>
 inline void Object<T>::translate( const VECTOR2_TYPE& ground_direction )
 {
    NodePath::set_x( NodePath::get_x() + ground_direction.get_x() );
    NodePath::set_y( NodePath::get_y() + ground_direction.get_y() );
 }
 
 /**
  * translate() translates this Object in the specified direction of the corresponding units.
  *
  * @param (const T&) x
  * @param (const T&) y
  * @param (const T&) z
  */
 template <typename T>
 inline void Object<T>::translate( const T& x, const T& y, const T& z )
 {
    NodePath::set_x( NodePath::get_x() + x );
    NodePath::set_y( NodePath::get_y() + y );
    NodePath::set_z( NodePath::get_z() + z );
 }
 
 /**
  * translate() translates this Object in the specified direction of the corresponding units.
  *
  * @param (const VECTOR3_TYPE&) ground_direction
  */
 template <typename T>
 inline void Object<T>::translate( const VECTOR3_TYPE& direction )
 {
    NodePath::set_x( NodePath::get_x() + direction.get_x() );
    NodePath::set_y( NodePath::get_y() + direction.get_y() );
    NodePath::set_z( NodePath::get_z() + direction.get_z() );
 }
 
 /** FRIEND FUNCTIONS **/
 
 #endif // OBJECT_H